
# Load necessary libraries
library(shiny)
library(drc)
library(readxl)
library(ggplot2)
library(shinythemes)

# Define the UI (User Interface)
ui <- fluidPage(
  theme = shinytheme("readable"),
  
  div(class = "jumbotron text-center",
      style = "background-color: #2c3e50; color: white; padding: 2em; border-radius: 10px;",
      
      h1("Kinetics Analysis App", windowTitle = "Kinetics Analysis App"),
      
      p("This Kinetics Analysis App allows you to visualize and analyse enzyme kinetics data using classic models including Michaelis-Menten, Eadie-Hofstee, Hanes-Woolf, and Lineweaver-Burke. Input your own experimental data or modify example datasets to fit kinetic curves and calculate key parameters like Vmax and Km."),
      
      p("If you encounter any issues, please contact Dr Claire Price via email: c.l.price@swansea.ac.uk")
  ),
  
  sidebarLayout(
    sidebarPanel(
      h4("Instructions"),
      p("1. Select the equation/model you want to analyze."),
      p("2. Edit the example data or input your own below."),
      p("3. Click 'Fit Curve' to generate the plot and view Vmax and Km."),
      tags$ul(
        tags$li(strong("Michaelis-Menten:"), " Non-linear model of enzyme kinetics."),
        tags$li(strong("Eadie-Hofstee:"), " Linearized form of enzyme kinetics."),
        tags$li(strong("Hanes-Woolf:"), " Plots [S]/v vs [S] for linear fitting."),
        tags$li(strong("Lineweaver-Burke:"), " Double reciprocal plot (1/v vs 1/[S]).")
      ),
      tags$hr(),
      h4("Choose Analysis Model"),
      selectInput("dataset", "Select the model:", 
                  choices = c("Michaelis-Menten" = "MM", 
                              "Eadie-Hofstee" = "EH", 
                              "Hanes-Woolf" = "HW", 
                              "Lineweaver-Burke" = "LB")),
      tags$hr(),
      h4("Example Data (Editable)"),
      p(strong("Instructions:"), "You can modify the example data below to explore how different values affect the model fitting."),
      p("Keep the two-line format:"),
      tags$ul(
        tags$li(strong("Substrate:")," followed by comma-separated numbers."),
        tags$li(strong("Reaction Rate:")," followed by comma-separated numbers")
      ),
    p("Do not remove the labels; both lines must be present for the app to work."),
    p("After editing the data or entering your own, press Fit Curve to generate the corresponding plot and see the estimated kinetic parameters (Vmax and Km)."), 
      br(), 
      textAreaInput("example_data", "Modify Example Data:",
                    value = "Substrate: 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 65, 70, 75, 80\nReaction Rate: 0.0286514, 0.034002, 0.0396636, 0.0454678, 0.0467814, 0.0489842, 0.050871, 0.0521924, 0.0545152, 0.0523106, 0.054696, 0.0541794, 0.0562896, 0.0559742, 0.0556756, 0.0563918"),
      tags$hr(),
      h4("Input Data"),
    p(strong("Instructions:")," To use your own data, enter comma-separated numbers in both fields below."),
    
    tags$ul(
      tags$li(strong("Substrate concentrations:"), " e.g., 5, 10, 15",),
      tags$li(strong("Reaction rates:"), " e.g., 0.01, 0.02, 0.03")
    ),
    p("Don't forget to check the box below to use this custom data instead of the example above."),
    p("Then click Fit Curve to run the analysis."),
    br(),
      checkboxInput("use_custom_data", strong("Use custom data instead of example"), FALSE),
    br(),
      textInput("substrate", "Substrate concentrations (comma-separated):", value = ""),
      textInput("reaction_rate", "Reaction rates (comma-separated):", value = ""),
      tags$hr(),
      actionButton("fit_curve", "Fit Curve", class = "btn btn-success btn-lg")
    ),
    
    mainPanel(
      h4("Generated Plot"),
      plotOutput("plotOutput", height = "400px"),
      tags$hr(),
      h4("Parameter Estimates"),
      verbatimTextOutput("parameters")
    )
  )
)

# Define the server logic
server <- function(input, output, session) {
  plot_reactive <- reactiveVal(NULL)
  
  output$exampleData <- renderText({
    input$example_data
  })
  
  observeEvent(input$fit_curve, {
    if (input$use_custom_data && input$substrate != "" && input$reaction_rate != "") {
      substrate <- as.numeric(unlist(strsplit(input$substrate, ",")))
      reaction_rate <- as.numeric(unlist(strsplit(input$reaction_rate, ",")))
    } else {
      example_lines <- strsplit(input$example_data, "\n")[[1]]
      substrate <- as.numeric(unlist(strsplit(sub("Substrate: ", "", example_lines[1]), ",")))
      reaction_rate <- as.numeric(unlist(strsplit(sub("Reaction Rate: ", "", example_lines[2]), ",")))
    }
    
    kineticsData <- data.frame(Substrate = substrate, dA = reaction_rate)
    plot <- NULL
    
    if (input$dataset == "MM") {
      mm <- drm(dA ~ Substrate, data = kineticsData, fct = MM.2())
      Vmax <- coef(mm)[1]
      Ks <- coef(mm)[2]
      output$parameters <- renderText({ paste0("Vmax: ", round(Vmax, 4), "
Km: ", round(Ks, 4)) })
      plot <- ggplot(kineticsData, aes(x = Substrate, y = dA)) +
        geom_point(color = "blue") +
        stat_smooth(method = "nls", formula = y ~ (Vmax * x) / (Ks + x), 
                    method.args = list(start = list(Vmax = Vmax, Ks = Ks)), se = FALSE, color = "red") +
        labs(title = "Michaelis-Menten Curve", x = "Substrate Concentration", y = "Reaction Rate") +
        theme_minimal()
      
    } else if (input$dataset == "EH") {
      r.velocity <- reaction_rate
      r.substrate <- reaction_rate / substrate
      lm_fit <- lm(r.velocity ~ r.substrate)
      Vmax <- coef(lm_fit)[1]
      Km <- -coef(lm_fit)[2]
      output$parameters <- renderText({ paste0("Vmax: ", round(Vmax, 4), "
Km: ", round(Km, 4)) })
      plot <- ggplot(data.frame(r.substrate, r.velocity), aes(x = r.substrate, y = r.velocity)) +
        geom_point(color = "blue") +
        geom_smooth(method = "lm", se = FALSE, color = "red") +
        labs(title = "Eadie-Hofstee Plot", x = "v/[S]", y = "v") +
        theme_minimal()
      
    } else if (input$dataset == "HW") {
      r.velocity <- substrate / reaction_rate
      lm_fit <- lm(r.velocity ~ substrate)
      output$parameters <- renderText({
        paste0("Vmax: ", round(1 / coef(lm_fit)[2], 4), 
               ", Km: ", round(coef(lm_fit)[1] / coef(lm_fit)[2], 4))
      })
      plot <- ggplot(data.frame(substrate, r.velocity), aes(x = substrate, y = r.velocity)) +
        geom_point(color = "blue") +
        geom_smooth(method = "lm", se = FALSE, color = "red") +
        labs(title = "Hanes-Woolf Plot", x = "[S]", y = "[S]/v") +
        theme_minimal()
      
    } else if (input$dataset == "LB") {
      r.substrate <- 1 / substrate
      r.velocity <- 1 / reaction_rate
      lineweaver.reg <- lm(r.velocity ~ r.substrate)
      plot <- ggplot(data.frame(r.substrate, r.velocity), aes(x = r.substrate, y = r.velocity)) +
        geom_point(color = "blue") +
        geom_abline(intercept = coef(lineweaver.reg)[1], slope = coef(lineweaver.reg)[2], color = "red") +
        labs(title = "Lineweaver-Burke Plot", x = "1/[S]", y = "1/v") +
        theme_minimal()
      intersection.y <- coef(lineweaver.reg)[1]
      Vmax <- 1 / intersection.y
      slope <- coef(lineweaver.reg)[2]
      intersection.x <- -(intersection.y / slope)
      Km <- -1 / intersection.x
      output$parameters <- renderText({
        paste0("Vmax: ", round(Vmax, 4), "
Km: ", round(Km, 4))
      })
    }
    
    plot_reactive(plot)
    output$plotOutput <- renderPlot({ plot })
  })
}



# Run the Shiny app
shinyApp(ui = ui, server = server)
